#if UNITY_2018_1_OR_NEWER && (UNITY_STANDALONE_OSX || UNITY_EDITOR_OSX)
using System.Runtime.InteropServices;
using UnityEngine;
using UnityEngine.Scripting;

using Unity.Collections;
using Unity.Collections.LowLevel.Unsafe;
using Unity.Jobs;

[assembly: AlwaysLinkAssembly]

namespace Unity.Barracuda
{

    [Preserve]
    public class MacBLAS : BLASPlugin
    {
        [DllImport("/System/Library/Frameworks/Accelerate.framework/Accelerate")]
        static extern unsafe void cblas_sgemm(CBLAS_ORDER __Order, CBLAS_TRANSPOSE __TransA, CBLAS_TRANSPOSE __TransB,
            int __M, int __N, int __K, float __alpha, float *__A, int __lda, float *__B, int __ldb,
            float __beta, float *__C, int __ldc);

        public bool IsNative()
        {
            return true;
        }

        public bool IsCurrentPlatformSupported()
        {
            return Application.platform == RuntimePlatform.OSXEditor ||
                   Application.platform == RuntimePlatform.OSXPlayer;
        }

        public unsafe void SGEMM(float* Ap, int AN, int AM, float* Bp, int BN, int BM, float* Cp, int CN, int CM,
            int bs,
            bool transposeA = false, bool transposeB = false)
        {
            cblas_sgemm(CBLAS_ORDER.CblasRowMajor, transposeA ? CBLAS_TRANSPOSE.CblasTrans : CBLAS_TRANSPOSE.CblasNoTrans,
                transposeB ? CBLAS_TRANSPOSE.CblasTrans : CBLAS_TRANSPOSE.CblasNoTrans,
                AN, BM, BN, 1.0f, Ap, AM, Bp, BM, 1.0f, Cp, CM);
        }

        public unsafe JobHandle ScheduleSGEMM(JobHandle dependsOn,
            float* Ap, int AN, int AM, float* Bp, int BN, int BM, float* Cp, int CN, int CM,
            int bs,
            bool transposeA = false, bool transposeB = false)
        {
            var job = new SGEMMJob();
            job.Ap = Ap; job.AN = AN; job.AM = AM;
            job.Bp = Bp; job.BN = BN; job.BM = BM;
            job.Cp = Cp; job.CN = CN; job.CM = CM;
            job.transposeA = transposeA;
            job.transposeB = transposeB;
            job.bs = bs;
            return job.Schedule(dependsOn);
        }

        unsafe struct SGEMMJob : IJob
        {
            [NativeDisableUnsafePtrRestriction][ReadOnly] public unsafe float* Ap;
            public int AN, AM;
            [NativeDisableUnsafePtrRestriction][ReadOnly] public unsafe float* Bp;
            public int BN, BM;
            [NativeDisableUnsafePtrRestriction]           public unsafe float* Cp;
            public int CN, CM;
            public int bs;
            public bool transposeA;
            public bool transposeB;

            public void Execute()
            {
                cblas_sgemm(CBLAS_ORDER.CblasRowMajor, transposeA ? CBLAS_TRANSPOSE.CblasTrans : CBLAS_TRANSPOSE.CblasNoTrans,
                    transposeB ? CBLAS_TRANSPOSE.CblasTrans : CBLAS_TRANSPOSE.CblasNoTrans,
                    AN, BM, BN, 1.0f, Ap, AM, Bp, BM, 1.0f, Cp, CM);
            }
        }

        internal enum CBLAS_ORDER
        {
            CblasRowMajor=101,
            CblasColMajor=102
        };

        internal enum CBLAS_TRANSPOSE
        {
            CblasNoTrans=111,
            CblasTrans=112,
            CblasConjTrans=113,
            AtlasConj=114
        };
    }
}
#endif // UNITY_OSX
